<?php

namespace wadeshuler\sms;

/**
 * SmsInterface is the interface that should be implemented by SMS classes.
 *
 * Usage Examples:
 *
 * ```php
 * Yii::$app->sms->compose('contact/sms', ['user' => $user])
 *     ->setFrom('12345')       // Your Twilio number (shortcode or full number)
 *     ->setTo('+15552224444')  // Full number including '+' and country code
 *     ->send();
 * ```
 *
 * -- or --
 *
 * ```php
 * Yii::$app->sms->compose()
 *     ->setFrom('12345')       // Your Twilio number (shortcode or full number)
 *     ->setTo('+15552224444')  // Full number including '+' and country code
 *     ->setMessage('Hello ' . $name . ', This is a test message!')
 *     ->send();
 * ```
 *
 */
interface SmsInterface
{
    /**
     * Creates a new message instance and optionally composes its body content via view rendering.
     *
     * @param string|null $view the view to be used for rendering the message body. This can be:
     *
     * - a string, which represents the view name or [path alias](https://www.yiiframework.com/doc/guide/2.0/en/concept-aliases)
     *   for rendering the text body of the message.
     *   In this case, the text body will be generated by applying `strip_tags()` to the body.
     * - null, meaning the message instance will be returned without body content.
     *
     * @param array $params the parameters (name-value pairs) that will be extracted and made available in the view file.
     * @return SmsInterface message instance.
     */
    public function compose($view = null, array $params = []);

    /**
     * Sends the given SMS message.
     * @param SmsInterface $message SMS message instance to be sent
     * @return bool whether the message has been sent successfully
     */
    public function send($message);

    /**
     * Sends multiple messages at once.
     *
     * This method may be implemented by some senders which support more efficient way of sending multiple messages in the same batch.
     *
     * @param array $messages list of SMS messages, which should be sent.
     * @return int number of messages that are successfully sent.
     */
    public function sendMultiple(array $messages);
}
